package gov.cms.grouper.snf.component.v100.logic;

import static gov.cms.grouper.snf.component.v100.TestUtil.of;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.spy;
import java.util.Arrays;
import java.util.List;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import gov.cms.grouper.snf.component.v100.TestUtil;
import gov.cms.grouper.snf.component.v100.logic.nursing.BscpLogic;
import gov.cms.grouper.snf.component.v100.logic.nursing.ReducedPhysicalFunctionLogic;
import gov.cms.grouper.snf.component.v100.logic.nursing.SpecialCare;
import gov.cms.grouper.snf.lego.SnfCache;
import gov.cms.grouper.snf.lego.SnfCache.Reference;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.enums.NursingCmg;
import gov.cms.grouper.snf.model.reader.Rai300;
import gov.cms.grouper.snf.util.ClaimInfo;

public class NursingLogicTest {

  public static final int version = 100;


  public static NursingLogic make(ClaimInfo claim, SpecialCare specialCare,
      ReducedPhysicalFunctionLogic physical, BscpLogic bscp) {

    if (specialCare == null) {
      specialCare = new SpecialCare(claim);
      specialCare = spy(specialCare);
    }

    Reference<ReducedPhysicalFunctionLogic> physicRef = SnfCache.of();

    if (bscp == null) {
      bscp = new BscpLogic(claim, physicRef);
      bscp = spy(bscp);
    }


    if (physical == null) {
      physical = new ReducedPhysicalFunctionLogic(claim, bscp);
      physical = spy(physical);
      physicRef.set(physical);
    }

    final NursingLogic logic = new NursingLogic(claim, specialCare, physical, bscp);
    NursingLogic spy = spy(logic);
    return spy;
  }

  public static NursingLogic make(List<Rai300> codedServices, int functionScore,
      NursingCmg clinicalComplexCategory) {
    ClaimInfo claim = of(functionScore);
    NursingLogic spy = make(claim, null, null, null);

    SpecialCare care = spy.getSpecialCare();
    doReturn(clinicalComplexCategory).when(care).evaluateClinicallyComplex(true);
    doReturn(codedServices).when(spy).getCodeServices();
    doReturn(clinicalComplexCategory).when(care).evaluateClinicallyComplex(true);
    doReturn(functionScore).when(claim).getFunctionScore();
    return spy;
  }



  @Test
  public void evaluateExtensiveServices() {

    NursingLogic nl = make(null, 0, null);
    Assertions.assertNull(nl.evaluateExtensiveServices());

    nl = make(Arrays.asList(), 0, null);
    Assertions.assertNull(nl.evaluateExtensiveServices());

    NursingCmg expectedCmg = NursingCmg.ES3;
    nl = make(Arrays.asList(Rai300.O0100M2, Rai300.O0100E2, Rai300.O0100F2), 0, null);
    NursingCmg actualCmg = nl.evaluateExtensiveServices();
    Assertions.assertEquals(expectedCmg, actualCmg);

    nl = make(Arrays.asList(Rai300.O0100M2, Rai300.O0100F2), 0, null);
    expectedCmg = NursingCmg.ES2;
    actualCmg = nl.evaluateExtensiveServices();
    Assertions.assertEquals(expectedCmg, actualCmg);

    nl = make(Arrays.asList(Rai300.O0100M2, Rai300.E0100A), 0, null);
    expectedCmg = NursingCmg.ES1;
    actualCmg = nl.evaluateExtensiveServices();
    Assertions.assertEquals(expectedCmg, actualCmg);

    nl = make(Arrays.asList(Rai300.E0100A, Rai300.J1550A), 0, null);
    Assertions.assertNull(nl.evaluateExtensiveServices());

    nl = make(Arrays.asList(Rai300.O0100M2, Rai300.E0100A), 15, NursingCmg.CA1);
    expectedCmg = NursingCmg.CA1;
    actualCmg = nl.evaluateExtensiveServices();
    Assertions.assertEquals(expectedCmg, actualCmg);

  }


  @Test
  public void testIsClassifiedBehavioralSymptomsCognitivePerformance() {
    ClaimInfo claim = of(0);
    Assertions.assertTrue(claim.isClassifiedBehavioralSymptomsCognitivePerformance(() -> 0, () -> 0,
        () -> 0, () -> true));
    Assertions.assertTrue(claim.isClassifiedBehavioralSymptomsCognitivePerformance(() -> 0, () -> 0,
        () -> 3, () -> false));
    Assertions.assertTrue(claim.isClassifiedBehavioralSymptomsCognitivePerformance(() -> 1, () -> 1,
        () -> 2, () -> false));
    Assertions.assertFalse(claim.isClassifiedBehavioralSymptomsCognitivePerformance(() -> 1,
        () -> 1, () -> 1, () -> false));
  }

  public static NursingLogic make(List<Assessment> assessments, int functionScore,
      NursingCmg specialCare, NursingCmg physical, NursingCmg extensive, NursingCmg behavioral) {
    if (assessments == null) {
      assessments = Arrays.asList();
    }
    ClaimInfo claim = of(assessments, functionScore);
    NursingLogic spy = make(claim, null, null, null);


    SpecialCare c = spy.getSpecialCare();
    doReturn(specialCare).when(c).exec();

    ReducedPhysicalFunctionLogic r = spy.getPhysical();
    doReturn(physical).when(r).exec();

    BscpLogic l = spy.getBscp();
    doReturn(behavioral).when(l).exec();
    doReturn(extensive).when(spy).evaluateExtensiveServices();

    return spy;
  }



  @Test
  public void testExec() {
    List<Assessment> assessments = Arrays.asList(new Assessment("O0100E2", "O0100E2", 1),
        new Assessment("O0100F2", "O0100F2", 1), new Assessment("O0100M2", "O0100M2", 1));

    NursingLogic com = make(assessments, 12, null, null, NursingCmg.ES3, null);
    Assertions.assertEquals(NursingCmg.ES3, com.exec());

    assessments = Arrays.asList(new Assessment("O0100E2", "O0100E2", 1),
        new Assessment("O0100F2", "O0100F2", 0), new Assessment("O0100M2", "O0100M2", 1));
    com = make(assessments, 12, null, null, NursingCmg.ES2, null);
    Assertions.assertEquals(NursingCmg.ES2, com.exec());

    assessments = Arrays.asList(new Assessment("O0100E2", "O0100E2", 0),
        new Assessment("O0100F2", "O0100F2", 0), new Assessment("O0100M2", "O0100M2", 1));
    com = make(assessments, 12, null, null, NursingCmg.ES1, null);
    Assertions.assertEquals(NursingCmg.ES1, com.exec());

    assessments = Arrays.asList(new Assessment("O0100E2", "O0100E2", 0),
        new Assessment("O0100F2", "O0100F2", 0), new Assessment("O0100M2", "O0100M2", 0));
    com = make(assessments, 12, NursingCmg.HDE1, null, null, null);
    Assertions.assertEquals(NursingCmg.HDE1, com.exec());

    assessments = Arrays.asList(new Assessment("O0100E2", "O0100E2", 0),
        new Assessment("O0100F2", "O0100F2", 0), new Assessment("O0100M2", "O0100M2", 0));
    com = make(assessments, 12, null, null, null, NursingCmg.PA1);
    Assertions.assertEquals(NursingCmg.PA1, com.exec());


    assessments = TestUtil.getAll(1);
    com = make(assessments, Integer.MIN_VALUE, null, null, NursingCmg.ES3, null);
    NursingCmg expected = NursingCmg.ES3;
    NursingCmg actual = com.exec();
    Assertions.assertEquals(expected, actual);
  }

}
